//BLUETOOTH.h

#define BLUETOOTH_API __declspec(dllexport)
/* Definitions and Structures used by the BTExplorer API                */

#ifndef __BTEXPAPISTRUCTS_H__
#define __BTEXPAPISTRUCTS_H__

#include "BTTypes.h"
#include "SDPAPI.h"

/* Name of BTExplorer's message queue.                                  */
#define BTEXP_QUEUE L"BTExpMsgQueue"

/* Name of BTExplorer's callback notification message queue.            */
#define BTEXP_CALLBACK_QUEUE L"BTExpCallbackMsgQueue"

/* Message Queue Commands                                               */
#define PING                              0

#define BTP_FIND_FIRST_DEVICE             1  
#define BTP_FIND_NEXT_DEVICE              2
#define BTP_FIND_DEVICE_CLOSE             3

#define BTP_FIND_FIRST_SERVICE            4
#define BTP_FIND_NEXT_SERVICE             5
#define BTP_FIND_SERVICE_CLOSE            6

#define BTP_FIND_FIRST_CONNECTION         7
#define BTP_FIND_NEXT_CONNECTION          8
#define BTP_FIND_CONNECTION_CLOSE         9
#define BTP_CREATE_CONNECTION             10
#define BTP_DELETE_CONNECTION             11
#define BTP_CONNECT                       12
#define BTP_DISCONNECT                    13

#define BTP_SET_AUTHENTICATION_CALLBACK   14
#define BTP_SET_CONNECTION_CALLBACK       15

#define BTP_FIND_LOCAL_DEVICE             16

#define BTP_SET_INCOMING_PIN              17
#define BTP_SET_OUTGOING_PIN              18

#define BTP_PERFORM_ACTION                19

#define BTP_SET_SECURITY_MODE             20
#define BTP_GET_SECURITY_MODE             21

#define BTP_SET_SCO_CONNECTION_STATE      22
#define BTP_GET_SCO_CONNECTION_STATE      23

/* Maximum message size allowed by the message queues.                  */
#define MAX_MSG_SIZE       1024

/* Maximum name length, including delimeter.                            */
#define BLUETOOTH_MAX_NAME_SIZE  (MAX_NAME_LENGTH + 1)

/* Error codes.                                                         */
#define BTP_ERROR                   0
#define BTP_ERROR_SUCCESS           1
#define BTP_ERROR_INVALID_PARAMETER 2
#define BTP_ERROR_INVALID_HANDLE    3
#define BTP_ERROR_NO_MORE           4
#define BTP_ERROR_MSG_SEND          5
#define BTP_ERROR_MSG_RECEIVE       6
#define BTP_ERROR_NO_COM_PORT       7
#define BTP_ERROR_BTEXP_NOT_RUNNING 8
#define BTP_ERROR_NO_KEY_AVAILABLE  9

/* Device states */
#define BTP_DEVICE_STATE_OFF 0
#define BTP_DEVICE_STATE_ON  1

/* SCO Connection States */
#define BTP_SCO_STATE_DISCONNECTED 0
#define BTP_SCO_STATE_CONNECTED    1

/* Enumeration of the supported profile types.                          */
typedef enum
{
	BTP_PROFILE_SPP,
	BTP_PROFILE_DUN,
	BTP_PROFILE_FAX,
	BTP_PROFILE_LAN,
	BTP_PROFILE_FILE_TRANSFER,
	BTP_PROFILE_HEADSET,
	BTP_PROFILE_HEADSET_AUDIO_GATEWAY,
	BTP_PROFILE_HANDS_FREE,
	BTP_PROFILE_HANDS_FREE_AUDIO_GATEWAY,
	BTP_PROFILE_HID_HOST,
	BTP_PROFILE_HID_DEVICE,
	BTP_PROFILE_UNKNOWN = -1
} BTP_Profile_Type;

typedef enum
{
	bdAll,
	bdHID
} BTP_DeviceType_t;

/* The following defines list the actions that can be used with the     */
/* BTP_Perform_Action API.                                              */

#define BTP_ACTION_SHOW_SETTINGS          (0x00000001UL)
/* Request BTExplorer to Show the Settings Dialog.                   */
/* Param1 = Not Used, Param2 = Not Used.                             */
/* Return1 = Not Used, Return2 = Not Used.                           */

#define BTP_ACTION_DELETE_ALL_DEVICES      (0x00000002UL)
/* Request BTExplorer to clear ALL currently remembered devices.     */
/* This is equivalent to the "Delete All Devices" UI option.         */
/* Param1 = Not Used, Param2 = Not Used.                             */
/* Return1 = Not Used, Return2 = Not Used.                           */

/************************************************************************/
/* Searching for remote devices                                         */
/************************************************************************/

/* Flags for the BTP_Device_Query_t and BTP_Device_Info_t structures.   */
/* This is a bit field with a single bit for each possible attribute.   */
/* Note that the ALL bit has been redefined although the code will still*/
/* support use of the old or new value of the ALL bit.  Any new builds  */
/* should use these new defines going forward.                          */
#define BTP_DEVICE_NONE             0x0001
#define BTP_DEVICE_AUTHENTICATED    0x0002
#define BTP_DEVICE_REMEMBERED       0x0004
#define BTP_DEVICE_CONNECTED        0x0008
#define BTP_DEVICE_ALL              0x8000

/* Enumeration handle for remote device queries.                        */
typedef HLOCAL BTP_Device_Find;

/* Structure specifying the criteria for device searches.               */
typedef struct _tagBTP_Device_Query_t
{
	Word_t DeviceAttributes;
	Byte_t InquiryTimeout;
} BTP_Device_Query_t;

/* Structure specifying the criteria for device searches.               */
typedef struct _tagBTP_Device_Query_Ex_t
{
	unsigned int      Size;
	Byte_t            Version;
	Word_t            DeviceAttributes;
	Byte_t            InquiryTimeout;
	BTP_DeviceType_t  DeviceType;
} BTP_Device_Query_Ex_t;

/* Structure containing information regarding a remote device.          */
typedef struct _tagBTP_Device_Info_t
{
	BD_ADDR_t         BD_ADDR;
	Class_of_Device_t ClassOfDevice;
	Word_t            DeviceAttributes;
	char              Name[BLUETOOTH_MAX_NAME_SIZE];
} BTP_Device_Info_t;

typedef struct _tagBTP_Device_Save_t
{
	BTP_Device_Query_Ex_t   DeviceQueryEx;
	unsigned int            index;
	unsigned int            numDevices;
	HLOCAL                  RemoteDeviceList;
} BTP_Device_Save_t;

/************************************************************************/
/* Searching for remote services                                        */
/************************************************************************/

/* Enumeration handle for remote service queries.                       */
typedef HLOCAL BTP_Service_Find;

/* Structure specifying criteria for service queries.                   */
typedef struct _tagBTP_Service_Query_t
{
	BD_ADDR_t BD_ADDR;
	Word_t NumberServiceUUID;
	SDP_UUID_Entry_t *Service;
} BTP_Service_Query_t;

/* Structure containing remote service or Favorite device information.  */
typedef struct _tagBTP_Service_Info_t
{
	BTP_Profile_Type  ProfileType;
	unsigned char     MajorVersion;
	unsigned char     MinorVersion;
	char              ServiceName[BLUETOOTH_MAX_NAME_SIZE];
	unsigned int      RFCOMMPort;
} BTP_Service_Info_t;

typedef struct _tagBTPSerialPortProfileInfo_t
{
	unsigned int RFCOMMServerPort;
	bool         UseActiveSync;
} BTPSerialPortProfileInfo_t;

typedef struct _tagBTPHIDProfileInfo_t
{
	unsigned int L2CAPControlChannel;
	unsigned int L2CAPInterruptChannel;
	Byte_t       DeviceSubclass;
	bool         VirtualCableSupported;
	bool         DeviceAutomaticReconnect;
	bool         DeviceNormallyConnectable;
} BTPHIDProfileInfo_t;

/* Structure containing remote service or Favorite device information.  */
typedef struct _tagBTP_Service_Info_Ex_t
{
	unsigned int      Size;
	unsigned int      Version;
	BTP_Profile_Type  ProfileType;
	unsigned char     MajorVersion;
	unsigned char     MinorVersion;
	char              ServiceName[BLUETOOTH_MAX_NAME_SIZE];
	unsigned int      ListIndex; //not used in version 2. For future enhancements
	union
	{
		BTPSerialPortProfileInfo_t RemoteSerialPortProfileInfo;
		BTPHIDProfileInfo_t        RemoteHIDProfileInfo;
	}ProfileInformation;
} BTP_Service_Info_Ex_t;

/* Structure used to store data between subsequent BTP_Find_Next_Service*/
/* calls.                                                               */
typedef struct _tagBTP_Service_Save_t
{
	BTP_Service_Query_t  ServiceQuery;
	unsigned int         index;
	unsigned int         numProfiles;
	HLOCAL               RemoteProfileList;
} BTP_Service_Save_t;


/************************************************************************/
/* Connections and Favorites                                            */
/************************************************************************/

/* Flags for the BTP_Connection_Query_t structure.                      */
#define BTP_CONNECTION_NONE         0
#define BTP_CONNECTION_REMEMBERED   1
#define BTP_CONNECTION_ACTIVE       2
#define BTP_CONNECTION_ALL          3

typedef HLOCAL BTP_Connection_Find;
typedef unsigned int BTP_Connection_ID;

/* Structure specifying search criteria for device searches.            */
typedef struct _tagBTP_Connection_Query_t
{
	Word_t ConnectionAttributes;
} BTP_Connection_Query_t;

/* Structure containing information about a Bluetooth connection.       */
typedef struct _tagBTP_Connection_Info_t
{
	BTP_Connection_ID    ConnectionID;
	BD_ADDR_t            BD_ADDR;
	unsigned int         RFCOMMPort;
	int                  LocalCOMPort;
	unsigned char        MajorVersion;
	unsigned char        MinorVersion;
	Word_t               ConnectionAttributes;
	BTP_Profile_Type     ProfileType;
} BTP_Connection_Info_t;

typedef struct _tagBTP_Connection_Info_Ex_t
{
	unsigned int         Size;
	unsigned int         Version;
	BTP_Connection_ID    ConnectionID;
	BD_ADDR_t            BD_ADDR;
	int                  LocalCOMPort;
	unsigned char        MajorVersion;
	unsigned char        MinorVersion;
	Word_t               ConnectionAttributes;
	BTP_Profile_Type     ProfileType;
	HLOCAL               ListHandle;// not used in version 2. For future enhancements.
	unsigned int         ListIndex;// not used in version 2. For future enhancements.
	union
	{
		BTPSerialPortProfileInfo_t RemoteSerialPortProfileInfo;
		BTPHIDProfileInfo_t        RemoteHIDProfileInfo;
	}ProfileInformation;
} BTP_Connection_Info_Ex_t;

/* Structure used to store data between subsequent                      */
/* BTP_Find_Next_Connection calls.                                      */
typedef struct _tagBTP_Connection_Save_t
{
	unsigned int           index1;
	unsigned int           index2;
	unsigned int           index3;
	unsigned int           numDevices;
	unsigned int           numFavorites;
	HLOCAL                 RemoteDeviceList;
	HLOCAL                 FavoriteDeviceList;
	BTP_Connection_Query_t ConnectionQuery;
} BTP_Connection_Save_t;

/* Structure used to store a security PIN to be used                    */
/* for authentication.                                                  */
typedef struct _tagBTP_PIN_Code_t
{
	Byte_t PIN_Code0;
	Byte_t PIN_Code1;
	Byte_t PIN_Code2;
	Byte_t PIN_Code3;
	Byte_t PIN_Code4;
	Byte_t PIN_Code5;
	Byte_t PIN_Code6;
	Byte_t PIN_Code7;
	Byte_t PIN_Code8;
	Byte_t PIN_Code9;
	Byte_t PIN_Code10;
	Byte_t PIN_Code11;
	Byte_t PIN_Code12;
	Byte_t PIN_Code13;
	Byte_t PIN_Code14;
	Byte_t PIN_Code15;
} BTP_PIN_Code_t;

/************************************************************************/
/* Callbacks                                                            */
/************************************************************************/

/* Authentication callback prototype.                                   */
typedef HRESULT (*BTP_Authentication_Callback_t) (const BD_ADDR_t *BD_ADDR,
												  char *const PassKey,
												  LPVOID CallbackParameter);

/* Structure containing all pertinent data associated with the          */
/* Authentication Callback so that a list of callbacks can be made.     */
typedef struct _tagBTP_Authentication_Callback_Data_t
{
	BD_ADDR_t                                      BD_ADDR;
	LPVOID                                         CallbackParameter;
	BTP_Authentication_Callback_t                  AuthenticationCallback;
	struct _tagBTP_Authentication_Callback_Data_t *NextAuthenticationCallbackData;
} BTP_Authentication_Callback_Data_t;

/* Connection Event callback prototype.                                 */
typedef void (*BTP_Connection_Callback_t) (const BTP_Connection_ID ConnectionID,
										   bool ConnectionState,
										   LPVOID CallbackParameter);

/* This structure allows for creating linked lists of Connection        */
/* callbacks.                                                           */
typedef struct _tagBTP_Connection_Callback_List_t
{
	BTP_Connection_Callback_t                  ConnectionCallback;
	LPVOID                                     CallbackParameter;
	struct _tagBTP_Connection_Callback_List_t *NextConnectionCallback;
} BTP_Connection_Callback_List_t;

/* Structure containing all pertinent data associated with the          */
/* Connection Callback so that a list of callbacks can be made.         */
typedef struct _tagBTP_Connection_Callback_Data_t
{
	BTP_Connection_ID                          ConnectionID;
	BTP_Connection_Callback_List_t            *ConnectionCallbackList;
	struct _tagBTP_Connection_Callback_Data_t *NextConnectionCallbackData;
} BTP_Connection_Callback_Data_t;

/* Enumeration of the callback types.                                   */
typedef enum
{
	BTP_AUTHENTICATION_CALLBACK,
	BTP_CONNECTION_CALLBACK
} BTP_Callback_Type;

/* Structure containing all data needed by CallbackNotifyThreadProc to  */
/* make the proper callback.                                            */
typedef struct _tagBTP_Callback_Notify_Data_t
{
	BTP_Callback_Type CallbackType;
	bool              ConnectionState;
	union
	{
		BD_ADDR_t         BD_ADDR;
		BTP_Connection_ID ConnectionID;
	} CallbackData;
} BTP_Callback_Notify_Data_t;

/* Structure containing all the data returned to BTExplorer by any of   */
/* the callbacks.                                                       */
typedef struct _tagBTP_Callback_Reply_Data_t
{
	bool Success;
	char Password[sizeof(PIN_Code_t)+1];
} BTP_Callback_Reply_Data_t;


/************************************************************************/
/* Outgoing Message Structures                                          */
/************************************************************************/

/* Device Find message information to BTExplorer                        */
typedef struct _tagBTP_Find_First_Device_To_BTE_t
{
	BTP_Device_Query_Ex_t DeviceQuery;
} BTP_Find_First_Device_To_BTE_t;

typedef struct _tagBTP_Find_Next_Device_To_BTE_t
{
	BTP_Device_Find DeviceFind;
} BTP_Find_Next_Device_To_BTE_t;

typedef struct _tagBTP_Find_Device_Close_To_BTE_t
{
	BTP_Device_Find DeviceFind;
} BTP_Find_Device_Close_To_BTE_t;

/* Service Find message information to BTExplorer                       */
/* * NOTE * The SDP UUID Entry array must be the last field so that it  */
/*          can be accessed like an array of SDP UUID Entries.          */
typedef struct _tagBTP_Find_First_Service_To_BTE_t
{
	BTP_Service_Query_t ServiceQuery;
	SDP_UUID_Entry_t    Service[1];
} BTP_Find_First_Service_To_BTE_t;

typedef struct _tagBTP_Find_Next_Service_To_BTE_t
{
	BTP_Service_Find ServiceFind;
} BTP_Find_Next_Service_To_BTE_t;

typedef struct _tagBTP_Find_Service_Close_To_BTE_t
{
	BTP_Service_Find ServiceFind;
} BTP_Find_Service_Close_To_BTE_t;


/* Connection Find message information to BTExplorer                    */
typedef struct _tagBTP_Find_First_Connection_To_BTE_t
{
	BTP_Connection_Query_t ConnectionQuery;
} BTP_Find_First_Connection_To_BTE_t;

typedef struct _tagBTP_Find_Next_Connection_To_BTE_t
{
	BTP_Connection_Find ConnectionFind;
} BTP_Find_Next_Connection_To_BTE_t;

typedef struct _tagBTP_Find_Connection_Close_To_BTE_t
{
	BTP_Connection_Find ConnectionFind;
} BTP_Find_Connection_Close_To_BTE_t;


/* Set Connection Callback message information to BTExplorer            */
typedef struct _tagBTP_Set_Connection_Callback_to_BTE_t
{
	BTP_Connection_ID ConnectionID;
	HANDLE            NotifyMsgQueue;
} BTP_Set_Connection_Callback_to_BTE_t;

/* Set Perform Action message information to BTExplorer                 */
typedef struct _tagBTP_Perform_Action_To_BTE_t
{   
	DWord_t            Action;
	DWord_t            Param1;
	DWord_t            Param2;
} BTP_Perform_Action_To_BTE_t;

/* Set PIN message to BTExplorer                                        */
typedef struct _tagBTP_PIN_t
{
	unsigned int   PINLength;
	BTP_PIN_Code_t PINCode;
} BTP_PIN_t;

typedef struct _tagBTP_Set_SCO_Connection_State_To_BTE_t
{
	BD_ADDR_t BD_ADDR;
	DWord_t   State;
} BTP_Set_SCO_Connection_State_To_BTE_t;

typedef enum
{
	BTP_SECURITYMODE_NONE,
	BTP_SECURITYMODE_AUTHENTICATE,
	BTP_SECURITYMODE_AUTHENTICATE_AND_ENCRYPT,
} BTP_Security_Mode_Type;

/* Set (and get) security mode message to BTExplorer                            */
typedef struct _tagBTP_Security_Mode_t
{
	BTP_Security_Mode_Type SecurityMode;
} BTP_Security_Mode_t;

/* Common Outgoing Message Structure                                    */
/* * NOTE * Any changes to this structure must also be reflected in the */
/*          header size defined below it.                               */
typedef struct _tagMessage_To_BTE_t
{
	int    Command;
	HANDLE ResponseQueueHandle;
	DWORD  ProcessID;
	union
	{
		BTP_Find_First_Device_To_BTE_t         FirstDeviceData;
		BTP_Find_Next_Device_To_BTE_t          NextDeviceData;
		BTP_Find_Device_Close_To_BTE_t         DeviceCloseData;
		BTP_Find_First_Service_To_BTE_t        FirstServiceData;
		BTP_Find_Next_Service_To_BTE_t         NextServiceData;
		BTP_Find_Service_Close_To_BTE_t        ServiceCloseData;
		BTP_Find_First_Connection_To_BTE_t     FirstConnectionData;
		BTP_Find_Next_Connection_To_BTE_t      NextConnectionData;
		BTP_Find_Connection_Close_To_BTE_t     ConnectionCloseData;
		BTP_Set_SCO_Connection_State_To_BTE_t  SCOConnectionData;
		BTP_Connection_Info_Ex_t               ConnectionInfo;
		BTP_Connection_ID                      ConnectionID;
		HANDLE                                 NotifyMsgQueue;
		BTP_Set_Connection_Callback_to_BTE_t   ConnectionCallbackData;
		BTP_PIN_t                              PINData;
		BTP_Perform_Action_To_BTE_t            ActionData;
		BD_ADDR_t                              BD_ADDR;
		BTP_Security_Mode_t                    SecurityModeData;
	} CommandData;
} Message_To_BTE_t;

#define MESSAGE_TO_BTE_HEADER_SIZE 12

/************************************************************************/
/* Incoming Message Structures                                          */
/************************************************************************/

/* Find local device message information from BTExplorer                */
typedef struct _tagBTP_Find_Local_Device_From_BTE_t
{
	BTP_Device_Info_t DeviceInfo;
} BTP_Find_Local_Device_From_BTE_t;

/* Find Device message information from BTExplorer.                     */
typedef struct _tagBTP_Find_First_Device_From_BTE_t
{
	BTP_Device_Find   DeviceFind;
	BTP_Device_Info_t DeviceInfo;
} BTP_Find_First_Device_From_BTE_t;

typedef struct _tagBTP_Find_Next_Device_From_BTE_t
{
	BTP_Device_Info_t DeviceInfo;
} BTP_Find_Next_Device_From_BTE_t;


/* Find Service message information from BTExplorer.                    */
typedef struct _tagBTP_Find_First_Service_From_BTE_t
{
	BTP_Service_Find      ServiceFind;
	BTP_Service_Info_Ex_t ServiceInfo;
} BTP_Find_First_Service_From_BTE_t;

typedef struct _tagBTP_Find_Next_Service_From_BTE_t
{
	BTP_Service_Info_Ex_t ServiceInfo;
} BTP_Find_Next_Service_From_BTE_t;


/* Find Connection message information from BTExplorer.                 */
typedef struct _tagBTP_Find_First_Connection_From_BTE_t
{
	BTP_Connection_Find      ConnectionFind;
	BTP_Connection_Info_Ex_t ConnectionInfo;
} BTP_Find_First_Connection_From_BTE_t;

typedef struct _tagBTP_Find_Next_Connection_From_BTE_t
{
	BTP_Connection_Info_Ex_t ConnectionInfo;
} BTP_Find_Next_Connection_From_BTE_t;

typedef struct _tagBTP_Perform_Action_From_BTE_t
{
	DWord_t Return1;
	DWord_t Return2;
} BTP_Perform_Action_From_BTE_t;

typedef struct _tagBTP_Get_SCO_Connection_State_From_BTE_t
{
	DWord_t State;
} BTP_Get_SCO_Connection_State_From_BTE_t;

/* Common Incoming Message Structure.                                   */
typedef struct _tagMessage_From_BTE_t
{
	HRESULT Result;
	union
	{
		BTP_Find_First_Device_From_BTE_t        FirstDeviceData;
		BTP_Find_Next_Device_From_BTE_t         NextDeviceData;
		BTP_Find_First_Service_From_BTE_t       FirstServiceData;
		BTP_Find_Next_Service_From_BTE_t        NextServiceData;
		BTP_Find_First_Connection_From_BTE_t    FirstConnectionData;
		BTP_Find_Next_Connection_From_BTE_t     NextConnectionData;
		BTP_Find_Local_Device_From_BTE_t        LocalDeviceData;
		BTP_Get_SCO_Connection_State_From_BTE_t SCOConnectionData;
		BTP_Connection_Info_Ex_t                ConnectionInfo;
		BTP_PIN_t                               PINData;
		BTP_Perform_Action_From_BTE_t           ActionResult;
		BTP_Security_Mode_t                     SecurityModeData;
	} CommandData;
}Message_From_BTE_t;

#define MESSAGE_FROM_BTE_HEADER_SIZE 4

#endif

class BLUETOOTH_API CBLUETOOTH{
public :
	CBLUETOOTH(void);
};

static FARPROC GetProcAddressWithCheck(HMODULE hModule, LPCWSTR lpProcName);
static bool GetProcAddresses(HMODULE hModule);

/* Initialization/Cleanup                                               */
EXTERN_C BLUETOOTH_API bool BLUETOOTH_Open();
EXTERN_C BLUETOOTH_API void BLUETOOTH_Close();

/* Informational                                                        */
EXTERN_C BLUETOOTH_API HRESULT BLUETOOTH_FindLocalDevice(BTP_Find_Local_Device_From_BTE_t *DeviceInfo);

/* Searching for Remote Devices                                         */
EXTERN_C BLUETOOTH_API HRESULT BLUETOOTH_FindFirstDevice(BTP_Device_Find *DeviceFind, BTP_Device_Info_t *DeviceInfo, const BTP_Device_Query_t *DeviceQuery);
EXTERN_C BLUETOOTH_API HRESULT BLUETOOTH_FindFirstDeviceEx(BTP_Device_Find *DeviceFind, BTP_Device_Info_t *DeviceInfo, const BTP_Device_Query_Ex_t *DeviceQuery);
EXTERN_C BLUETOOTH_API HRESULT BLUETOOTH_FindNextDevice(BTP_Device_Find DeviceFind, BTP_Device_Info_t *DeviceInfo);
EXTERN_C BLUETOOTH_API HRESULT BLUETOOTH_FindDeviceClose(BTP_Device_Find DeviceFind);

/* Searching for Remote Services                                        */
EXTERN_C BLUETOOTH_API HRESULT BLUETOOTH_FindFirstService(BTP_Service_Find *ServiceFind, BTP_Service_Info_t *ServiceInfo, const BTP_Service_Query_t *ServiceQuery);
EXTERN_C BLUETOOTH_API HRESULT BLUETOOTH_FindFirstServiceEx(BTP_Service_Find *ServiceFind, BTP_Service_Info_Ex_t *ServiceInfo, const BTP_Service_Query_t *ServiceQuery);
EXTERN_C BLUETOOTH_API HRESULT BLUETOOTH_FindNextService(BTP_Service_Find ServiceFind, BTP_Service_Info_t *ServiceInfo);
EXTERN_C BLUETOOTH_API HRESULT BLUETOOTH_FindNextServiceEx(BTP_Service_Find ServiceFind, BTP_Service_Info_Ex_t *ServiceInfo);
EXTERN_C BLUETOOTH_API HRESULT BLUETOOTH_FindServiceClose(BTP_Service_Find ServiceFind);

/* Connections and Favorites                                            */
EXTERN_C BLUETOOTH_API HRESULT BLUETOOTH_FindFirstConnection(BTP_Connection_Find *ConnectionFind, BTP_Connection_Info_t *ConnectionInfo, const BTP_Connection_Query_t *ConnectionQuery);
EXTERN_C BLUETOOTH_API HRESULT BLUETOOTH_FindFirstConnectionEx(BTP_Connection_Find *ConnectionFind, BTP_Connection_Info_Ex_t *ConnectionInfoEx, const BTP_Connection_Query_t *ConnectionQuery);
EXTERN_C BLUETOOTH_API HRESULT BLUETOOTH_FindNextConnection(BTP_Connection_Find ConnectionFind, BTP_Connection_Info_t *ConnectionInfo);
EXTERN_C BLUETOOTH_API HRESULT BLUETOOTH_FindNextConnectionEx(BTP_Connection_Find ConnectionFind, BTP_Connection_Info_Ex_t *ConnectionInfo);
EXTERN_C BLUETOOTH_API HRESULT BLUETOOTH_FindConnectionClose(BTP_Connection_Find ConnectionFind);

EXTERN_C BLUETOOTH_API HRESULT BLUETOOTH_CreateConnection(BTP_Connection_Info_t *ConnectionInfo);
EXTERN_C BLUETOOTH_API HRESULT BLUETOOTH_CreateConnectionEx(BTP_Connection_Info_Ex_t *ConnectionInfo);
EXTERN_C BLUETOOTH_API HRESULT BLUETOOTH_DeleteConnection(BTP_Connection_ID ConnectionID);

EXTERN_C BLUETOOTH_API HRESULT BLUETOOTH_Connect(BTP_Connection_ID ConnectionID);
EXTERN_C BLUETOOTH_API HRESULT BLUETOOTH_Disconnect(BTP_Connection_ID ConnectionID);

EXTERN_C BLUETOOTH_API HRESULT BLUETOOTH_SetIncomingPIN(BTP_PIN_t *NewPIN, BTP_PIN_t *OldPIN);
EXTERN_C BLUETOOTH_API HRESULT BLUETOOTH_SetOutgoingPIN(BTP_PIN_t *NewPIN, BTP_PIN_t *OldPIN);

EXTERN_C BLUETOOTH_API HRESULT BLUETOOTH_PerformAction(DWord_t Action, DWord_t Param1, DWord_t Param2, DWord_t *Return1, DWord_t *Return2);

EXTERN_C BLUETOOTH_API HRESULT BLUETOOTH_SetBluetoothState(DWord_t BluetoothState);
EXTERN_C BLUETOOTH_API HRESULT BLUETOOTH_GetBluetoothState(DWord_t *BluetoothState);

EXTERN_C BLUETOOTH_API HRESULT BLUETOOTH_SetSCOConnectionState(BD_ADDR_t *BD_ADDR, DWord_t ConnectionState);
EXTERN_C BLUETOOTH_API HRESULT BLUETOOTH_GetSCOConnectionState(BD_ADDR_t *BD_ADDR, DWord_t *ConnectionState);

EXTERN_C BLUETOOTH_API HRESULT BLUETOOTH_SetSecurityMode(BTP_Security_Mode_Type SecurityMode);
EXTERN_C BLUETOOTH_API HRESULT BLUETOOTH_GetSecurityMode(BTP_Security_Mode_Type *SecurityMode);

/* Callback Functions                                                   */
EXTERN_C BLUETOOTH_API HRESULT BLUETOOTH_SetAuthenticationCallback(const BD_ADDR_t *BD_ADDR, BTP_Authentication_Callback_t AuthenticationCallback, LPVOID CallbackParameter);
EXTERN_C BLUETOOTH_API HRESULT BLUETOOTH_SetConnectionCallback(BTP_Connection_ID ConnectionID, BTP_Connection_Callback_t ConnectionCallback, LPVOID CallbackParameter);
EXTERN_C BLUETOOTH_API HRESULT BLUETOOTH_Authentication_Callback(const BD_ADDR_t *BD_ADDR, char *const PassKey, LPVOID CallbackParameter);
EXTERN_C BLUETOOTH_API void	BLUETOOTH_Connection_Callback(const BTP_Connection_ID ConnectionID, bool ConnectionState, LPVOID CallbackParameter);

EXTERN_C BLUETOOTH_API BOOL BLUETOOTH_GetVersionInfo(TCHAR *pszVersion);

