/*-------------------------------------------------------------------------*
 *                                                                         *
 *   THIS IS AN UNPUBLISHED WORK CONTAINING CONFIDENTIAL AND PROPRIETARY   *
 *   INFORMATION.  IF PUBLICATION OCCURS, THE FOLLOWING NOTICE APPLIES:    *
 *        "COPYRIGHT 2012 AMARULA SOLUTIONS, ALL RIGHTS RESERVED"          *
 *                                                                         *
 *   NO DERIVATIVE WORKS ARE ALLOWED WITHOUT OUR EXPLICIT CONSENSOUS.      *
 *                                                                         *
 *   Amarula Solutions BV                                Cruquiuskade 47   *
 *   www.amarulasolutions.com                       Amsterdam 1018 AM NL   *
 *                                                                         *
 *-------------------------------------------------------------------------*/

package com.amarula.barcodescanner.sdk;

import android.content.ComponentName;
import android.content.Context;
import android.content.Intent;
import android.content.ServiceConnection;
import android.os.IBinder;
import android.os.RemoteException;
import android.util.Log;


import com.amarula.barcodescanner.sdk.IBarcodeScannerService;

/**
 * <p>Manages the connection to {@link BarcodeScannerService}.</p>
 * <p>{@link bindToBarcodeScannerService()} start the connection process (bind) to {@link BarcodeScannerService}
 * while {@link unbindFromBarcodeScannerService()} disconnect from it.</p>
 * <p>While connected you can use:</p>
 * <ul>
 * <li>{@link scanOneBarcode()}: To trigger the scanning of a single barcode</li>
 * <li>{@link scanMultipleBarcodes()}: To trigger the scanning of multiple barcodes (Service doesen't stop scanning after the first positive result)</li>
 * <li>{@link stopScanning()}: To abort the current scanning</li>
 * </ul>
 *
 * <p>Bind to the barcode scanner service only if needed: the service keeps "warm"
 * the hardware for scanning from binding till unbind consuming more power
 * so Applications must to bind to the scanner only when needed:
 * <ul>
 * <li>If connecting from an Activity, bind on onResume(), unbind on onPause()</li>
 * <li>If connecting from a Service, bind on need and unbind on Screen Off</li>
 * </ul>
 *
 * <p>Since the preparation and release of {@link BarcodeScannerService} are long time tasks, you can use
 * {@link startCachedBarcodeScannerService()} to bind to the scannig service.<br/>
 * Doing this, {@link BarcodeScannerService} will be started to cache it's state with a
 * timeout and {@link IBarcodeScannerService.release()} will not be called
 * on unbind. In this way it is the service
 * itself that is in charge of releasing it's state when no clients are connected after a timeout.
 * </p>
 *
 * @author Alberto Panizzo alberto@amarulasolutions.com
 *
 */
public class BarcodeScannerConnection {

	private static String TAG = "BarcodeScannerConnection";

	/** The primary interface we will be calling on the service. */
	private IBarcodeScannerService mService = null;

	private boolean mIsBound;
	private Context mContext;
	private Object mLock = new Object();
	private BarcodeScannerConnectionInterface mCallback = null;

	public BarcodeScannerConnection(Context ctx, BarcodeScannerConnectionInterface cb) {
		mContext = ctx;
		mCallback = cb;
	}

	/**
	 * Class for interacting with the main interface of the service.
	 */
	private ServiceConnection mConnection = new ServiceConnection() {
		@Override
        public void onServiceConnected(ComponentName className,
				IBinder service) {
			synchronized(mLock) {
				mService = IBarcodeScannerService.Stub.asInterface(service);
				Log.d(TAG, "Connected to the BarcodeScanner service");
			}
			try {
				mCallback.onServiceConnected(mService.getStatus());
			} catch (RemoteException e) {
				e.printStackTrace();
			}
		}

		@Override
        public void onServiceDisconnected(ComponentName className) {
			// This is called when the connection with the service has been
			// unexpectedly disconnected -- that is, its process crashed.
			synchronized(mLock) {
				mService = null;
				mIsBound = false;
			}
			mCallback.onServiceDisconnected();
		}
	};

	public void bindToBarcodeScannerService() {
		synchronized(mLock) {
			if (mIsBound && mService != null)
				return;
			Log.d(TAG, "Binding to the service..");
			mContext.bindService(new Intent(BarcodeScanner.BARCODESCANNER_ACTION),
					mConnection, Context.BIND_AUTO_CREATE);
			mIsBound = true;
		}
	}

	public void startCachedBarcodeScannerService() {
		synchronized(mLock) {
			Intent cachedBarcodeScannerServiceIntent = new Intent(BarcodeScanner.BARCODESCANNER_ACTION);
			cachedBarcodeScannerServiceIntent.addCategory(BarcodeScanner.BARCODESCANNER_CACHE_STATE_CATEGORY);
			mContext.startService(cachedBarcodeScannerServiceIntent);
		}
		bindToBarcodeScannerService();
	}

	public void unbindFromBarcodeScannerService() {
		synchronized(mLock) {
			if (mIsBound) {
				Log.d(TAG, "Unbinding from the service..");
				mContext.unbindService(mConnection);
				mIsBound = false;
			}
		}
	}

	public void stopScanning() {
		synchronized(mLock) {
			if (mService != null) {
				try {
					mService.stopReading();
				} catch (RemoteException e) {
					e.printStackTrace();
				}
				Log.d(TAG, "Stop scanning");
			} else {
				Log.w(TAG, "stopScanning(): Service still not connected");
			}
		}
	}

	public int scanMultipleBarcodes() {
		int ret = -1;
		synchronized(mLock) {
			if (mService != null) {
				try {
					ret = mService.readBarcodeMulti();
				} catch (RemoteException e) {
					e.printStackTrace();
				}
				Log.d(TAG, "Read multiple barcodes");
			} else {
				Log.w(TAG, "scanMultipleBarcodes(): Service still not connected");
			}
		}
		return ret;
	}

	public int scanOneBarcode() {
		int ret = -1;
		synchronized(mLock) {
			if (mService != null) {
				try {
					ret = mService.readBarcodeSingle();
				} catch (RemoteException e) {
					e.printStackTrace();
				}
				Log.d(TAG, "Read a single barcode");
			} else {
				Log.w(TAG, "scanOneBarcode(): Service still not connected");
			}
		}
		return ret;
	}

    public boolean registerServerCallback(IBarcodeServiceCallback cb) {
        boolean ret = false;
        if (mService != null) {
            try {
                ret = mService.registerCallback(cb);
            } catch (RemoteException e) {
                e.printStackTrace();
            }
            Log.d(TAG, "Registering Callback");
        } else {
            Log.w(TAG, "registerServerCallback(): Service still not connected");
        }
        return ret;
    }

    public boolean unregisterServerCallback(IBarcodeServiceCallback cb) {
        boolean ret = false;
        if (mService != null) {
            try {
                ret = mService.unregisterCallback(cb);
            } catch (RemoteException e) {
                e.printStackTrace();
            }
            Log.d(TAG, "Unregistering Callback");
        } else {
            Log.w(TAG, "unregisterServerCallback(): Service still not connected");
        }
        return ret;
    }

    public void setEnableIDC(boolean enable){
        Log.d(TAG, "setEnableIDC called in BarcodeScannerConnection");
        synchronized(mLock) {
            try {
                mService.setEnableIDC(enable);
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
    }

    public void setParameter(int key, int value){
        synchronized(mLock) {
            try {
                mService.setParameter(key, value);
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
    }
}
