//
//  LQSandbox.swift
//  DuanZi
//
//  Created by lequwuxian1 on 2019/1/8.
//  Copyright © 2019年 lequwuxian. All rights reserved.
//

import UIKit

/// 沙盒目录
class LQSandbox: NSObject {
    
    /// 文档目录，需要ITUNES同步备份的数据存这里
    ///
    /// - Returns: <#return value description#>
    public class func docPath() -> String {
        return NSSearchPathForDirectoriesInDomains(FileManager.SearchPathDirectory.documentDirectory, FileManager.SearchPathDomainMask.userDomainMask, true).first!
    }
    
    
    /// 配置目录，配置文件存这里
    ///
    /// - Returns: <#return value description#>
    public class func libPrefPath() -> String {
        let paths = NSSearchPathForDirectoriesInDomains(.libraryDirectory, .userDomainMask, true)
        let path = paths.first?.appending("/Preference")
        _ = self.touch(path: path!)
        return path!
    }
    
    ///  缓存目录，系统在磁盘空间不足的情况下会删除里面的文件，ITUNES会删除
    public class func libCachePath() -> String {
        let paths = NSSearchPathForDirectoriesInDomains(.libraryDirectory, .userDomainMask, true)
        let path = paths.first?.appending("/Caches")
        _ = self.touch(path: path!)
        return path!
    }
    
    ///  缓存目录，APP退出后，系统可能会删除这里的内容
    public class func tmpPath() -> String {
        let path = NSTemporaryDirectory()
        _ = self.touch(path: path)
        return path
    }
    
    
    /// 文件是否存在
    /// - Parameter path: <#path description#>
    /// - Returns: <#description#>
    public static func fileExists(path: String) -> Bool {
        return FileManager.default.fileExists(atPath: path)
    }
    
    /// check文件目录路径，如果没有则创建
    ///
    /// - Parameter path: <#path description#>
    /// - Returns: <#return value description#>
    @discardableResult
    public class func touch(path: String) -> Bool {
        if FileManager.default.fileExists(atPath: path) == false {
            do {
                try FileManager.default.createDirectory(atPath: path, withIntermediateDirectories: true, attributes: nil)
                return true
            } catch {}
            return false
        }
        return true
    }
    
    /// check文件路径，如果没有则创建
    public class func touch(file: String) -> Bool{
        if false == FileManager.default.fileExists(atPath: file) {
            return FileManager.default.createFile(atPath: file,
                                                  contents: Data(),
                                                  attributes: nil)
        }
        return true
    }
    
    
    /// 创建目录
    ///
    /// - Parameter path: 目录路径
    public class func createDirectory(atPath path: String){
        if false == FileManager.default.fileExists(atPath: path, isDirectory: nil) {
            _ = self.touch(path: path)
        }
    }
    
    /// 返回目录下所有给定后缀的文件
    ///
    /// - Parameters:
    ///   - path: 目录绝对路径
    ///   - type: 文件后缀名(".txt")
    ///   - operation: 预留参数
    public class func allFiles(atPath path: String, fileType type: String, operation: Int) -> [String]{
        var paths = Array<String>()
        var temps : Array<String>?
        do {
            temps = try FileManager.default.contentsOfDirectory(atPath: path)
        }catch{  }
        
        for fileName in temps! {
            var isDir : ObjCBool = true
            let fullPath = NSString(string: path).appendingPathComponent(fileName)
            
            if FileManager.default.fileExists(atPath: fullPath, isDirectory: &isDir) {
                if isDir.boolValue == false && fileName.hasSuffix(type) && fileName.hasPrefix("._") == false && fileName.hasPrefix("./") == false { //过滤掉隐藏文件
                    paths.append(fullPath)
                } else if (isDir.boolValue == true) {
                    let subPaths = allFiles(atPath: fullPath, fileType: type, operation: operation)
                    paths.append(contentsOf: subPaths)
                }
            }
        }
        return paths
    }
    
    /// 返回目录文件的size,单位字节
    ///
    /// - Parameters:
    ///   - path: 目录路径
    ///   - disk: 是否是磁盘占用的size
    /// - Returns: <#return value description#>
    public class func size(atPath path: String = "", diskModel disk:Bool) -> UInt64{
        var totalSize : UInt64 = 0
        if path.count == 0 {
            return totalSize
        }
        var searchPaths = [path]
        
        while searchPaths.count > 0 {
            
            autoreleasepool{
                let fullPath = searchPaths.first
                searchPaths.remove(at: 0)
                
                var fileStat  = stat()
                
                if lstat(NSString(string: fullPath!).fileSystemRepresentation, &fileStat) == 0 {
                    
                    if (fileStat.st_mode&S_IFDIR)>0 {
                        var childSubPaths : [String]?
                        
                        do{
                            childSubPaths = try FileManager.default.contentsOfDirectory(atPath: fullPath!)
                        }catch{}
                        
                        for childItem in childSubPaths!{
                            let childPath = NSString(string: fullPath!).appendingPathComponent(childItem)
                            childSubPaths?.insert(childPath, at: 0)
                        }
                    }else{
                        if disk {
                            totalSize += UInt64(fileStat.st_blocks * 512)
                        }else{
                            totalSize += UInt64(fileStat.st_size)
                        }
                    }
                }
            }
        }
        
        return totalSize
    }
    
    
    /// 设置目录里的文件不备份
    ///
    /// - Parameter url: <#url description#>
    /// - Returns: <#return value description#>
    public class func skipFileBackupForItem(atURL url:URL) -> Bool {
        if FileManager.default.fileExists(atPath: url.path) == false {
            return false
        }

        let filePath = NSString(string: url.path).fileSystemRepresentation
        let attrName = "com.apple.MobileBackup"
        var attrValue : UInt8 = 1
        
        return setxattr(filePath, attrName, &attrValue, MemoryLayout.size(ofValue: attrValue), 0, 0) == 0
    }

    /// 如果文件为文件夹，删除该文件夹下所有内容,如果路径为文件，则删除该文件
    ///
    /// - Parameter path: <#path description#>
    /// - Returns: <#return value description#>
    public class func removeAllItem(inPath path: String) ->Bool{
        var isDir : ObjCBool = true
        
        if false == FileManager.default.fileExists(atPath: path, isDirectory: &isDir) {
            return true
        }
    
        if isDir.boolValue == false && FileManager.default.isDeletableFile(atPath: path) {
            do {
                try FileManager.default.removeItem(atPath: path)
                return true
            }catch {}
            return false
        }
        var contents : [String]?
        
        do {
            contents = try FileManager.default.contentsOfDirectory(atPath: path)
        }catch {}
        
        let e = NSArray(array: contents!).objectEnumerator()
        var filename : String? = e.nextObject() as? String
        
        while filename != nil {
            let  subPath = NSString(string: path).appendingPathComponent(filename!)
            
            if FileManager.default.isDeletableFile(atPath: subPath){
                do {
                    try FileManager.default.removeItem(atPath: subPath)
                }catch {}
            }
            filename = e.nextObject() as? String
        }
        return true
    }
    
}
