//
//  ViewController.swift
//  Gloves
//
//  Created by ifeng on 2021/11/27.
//


import UIKit
import CoreBluetooth


let BELWriteDataServiceUUID = "FFE0"
let BELWriteDataCharacteristicUUID = "FFE1"



class ViewController: UIViewController {
    var centralManager:CBCentralManager!
    var connectPeripheral:CBPeripheral?

    var svc = SViewController()
    
    var tableView:UITableView!
    
    var dataList:[CBPeripheral] = []
    
    var instruction :  Instruction?    
    
    var powerFirst = YES
    
    var powerIsA4 = NO
    var powerIds:[String] {
        if powerIsA4 {
            return [Instruction.ID.power1500, Instruction.ID.power2000]
        }
        return [Instruction.ID.power500, Instruction.ID.power1000, Instruction.ID.power1500, Instruction.ID.power2000, Instruction.ID.power2300, Instruction.ID.power2700, Instruction.ID.power3000]
    }
    var powerTitles:[String] {
        if powerIsA4 {
            return ["功率 1500".localization, "功率 2000".localization]
        }
        return ["功率 500".localization, "功率 1000".localization, "功率 1500".localization, "功率 2000".localization, "功率 2300".localization, "功率 2700".localization, "功率 3000".localization]
    }
    

    override func viewDidLoad() {
        super.viewDidLoad()
        view.backgroundColor = .white
        
        initUI()

    }
    
    func initUI() {
        centralManager = CBCentralManager(delegate: self, queue: nil)
        
        
        let searchButton = UIButton(type: .custom)
        searchButton.setTitle("搜索蓝牙".localization, for: .normal)
        searchButton.titleLabel?.font = .lqsSmaller
        searchButton.backgroundColor = .lqsMain
        searchButton.setTitleColor(.white, for: .normal)
        searchButton.setTitleColor(.lqsCancle, for: .highlighted)
        searchButton.frame = CGRect(x: 15, y: 20 + kLQStatusBarHeight, width: 130, height: 42)
        searchButton.addTarget(self, action: #selector(searchBL), for: .touchUpInside)
        searchButton.round(radius: 21)
        view.addSubview(searchButton)
        
        let stopSearchButton = UIButton(type: .custom)
        stopSearchButton.setTitle("停止搜索".localization, for: .normal)
        stopSearchButton.backgroundColor = .lqsMain
        stopSearchButton.titleLabel?.font = .lqsSmaller
        stopSearchButton.frame = CGRect(x:  view.frame.width * 0.5 + 15, y: searchButton.frame.minY, width: 130, height: 42)
        stopSearchButton.addTarget(self, action: #selector(stopSearchBL), for: .touchUpInside)
        stopSearchButton.round(radius: 21)
        stopSearchButton.setTitleColor(.white, for: .normal)
        stopSearchButton.setTitleColor(.lqsCancle, for: .highlighted)
        view.addSubview(stopSearchButton)
        
        
        tableView = UITableView(frame: CGRect(x: 15, y: searchButton.frame.maxY + 30, width: view.frame.width - 30, height: 300), style: .plain)
        tableView.delegate = self
        tableView.dataSource = self
        tableView.setCornerRadius(radius: 8, borderColor: .lqsSpaceLine, borderWidth: 1)
        view.addSubview(tableView)
        
        let logButton = UIButton(type: .custom)
        logButton.setTitle("日志".localization, for: .normal)
        logButton.frame = CGRect(x: view.frame.width - 60, y: view.frame.height - 100, width: 50, height: 50)
        logButton.backgroundColor = .lqsMain
        logButton.addTarget(self, action: #selector(showLogs(_:)), for: .touchUpInside)
        view.addSubview(logButton)

        
        svc.powerButton.isEnabled = NO
        svc.sessionModelButton.addTarget(self, action: #selector(setSessionModel(_:)), for: .touchUpInside)
        svc.powerButton.addTarget(self, action: #selector(setPower(_:)), for: .touchUpInside)
        svc.pdButton.addTarget(self, action: #selector(setPD(_:)), for: .touchUpInside)
        svc.inventoryButton.addTarget(self, action: #selector(startInventory(_:)), for: .touchUpInside)
        svc.disconnectButton.addTarget(self, action: #selector(disconnectButtonAction(_:)), for: .touchUpInside)
    }
    
    
    @objc func searchBL() {
        let service = CBUUID(string: "0000ffe0-0000-1000-8000-00805f9b34fb")
        centralManager.scanForPeripherals(withServices: [service], options: [CBCentralManagerScanOptionAllowDuplicatesKey: NSNumber(value: YES)])
    }
    
    @objc func stopSearchBL() {
        if connectPeripheral !=  nil {
            centralManager.cancelPeripheralConnection(connectPeripheral!)
            connectPeripheral = nil
        }
        centralManager.stopScan()
        dataList.removeAll()
        tableView.reloadData()
    }
    
    
    @objc  func setSessionModel(_ sender:UIButton) {
        let titles = ["会话模式0".localization, "会话模式1".localization]
        let ids = [Instruction.ID.sessionModel0, Instruction.ID.sessionModel1]
        svc.sheetShow(withTitle: "设置会话模式".localization, message: nil, cancel: "取消".localization, others:  titles) {[weak self] idx in
            if self == nil { return }
            
            let ins = Instruction(ids[idx])
            if self!.send(ins) {
                sender.setTitle("设置会话中...".localization, for: .normal)
            }
            
        }
        
    }
    
    @objc  func setPower(_ sender:UIButton) {
        let titles = powerTitles
        let ids = powerIds
        svc.sheetShow(withTitle: "设置功率".localization, message: nil, cancel: "取消".localization, others:  titles) {[weak self] idx in
            if self == nil { return }
            
            let ins = Instruction(ids[idx])
            if self!.send(ins) {
                sender.setTitle("设置功率中...".localization, for: .normal)
            }
            
        }
        
    }
    
    @objc func setPD(_ sender:UIButton) {
        let titles = ["频段：中国".localization, "频段：北美".localization, "频段：欧洲".localization]
        let ids = [Instruction.ID.pdChina, Instruction.ID.pdBeimei, Instruction.ID.pdOzhou]
        svc.sheetShow(withTitle: "设置频段".localization, message: nil, cancel: "取消".localization, others:  titles) {[weak self] idx in
            if self == nil { return }
            
            let ins = Instruction(ids[idx])
            if self!.send(ins) {
                sender.setTitle("设置频段中...".localization, for: .normal)
            }
            
        }
    }
    

    @objc func startInventory(_ sender:UIButton) {
        svc.isPanding = YES
        sender.isEnabled = NO
        sender.backgroundColor = .lqsCancle
        let ins = Instruction(Instruction.ID.inventory)
        send(ins)
    }
    
    @objc func disconnectButtonAction(_ sender:UIButton) {
        svc.isPanding = NO
        DispatchQueue.main.asyncAfter(deadline: .now() + .milliseconds(500)) {
            self.svc.tagsMap.removeAll()
            self.svc.tagsTableView.reloadData()
            self.svc.dismiss(animated: YES, completion: nil)
            if self.connectPeripheral !=  nil {
                self.centralManager.cancelPeripheralConnection(self.connectPeripheral!)
                self.connectPeripheral = nil
                self.tableView.reloadData()
            }
        }
        
    }
    
    
    @objc func showLogs(_ sender:UIButton) {
        let vc = LogsViewController()
        vc.dissmissComplet = {[weak self] url in
            let viewCtrl = UIActivityViewController(activityItems: [url], applicationActivities: nil)
            viewCtrl.completionWithItemsHandler = {(type,completed,_,activityError)in
                md_log("")
            }
            self?.present(viewCtrl, animated: YES, completion: nil)

        }
        present(vc, animated: YES, completion: nil)
    }

}

extension ViewController : CBCentralManagerDelegate {
    func centralManagerDidUpdateState(_ central: CBCentralManager) {
                switch central.state {
                case .unknown:
                    md_log("蓝牙功能错误")
                    LQJargon.hiddenJargon(message: ("蓝牙功能错误".localization))
                    break
                case .resetting:
                    md_log("蓝牙功能失去连接")
                    LQJargon.hiddenJargon(message: ("蓝牙功能失去连接".localization))
                    break
                case .unsupported:
                    md_log("蓝牙功能不支持")
                    LQJargon.hiddenJargon(message: ("蓝牙功能不支持".localization))
                    break
                case .unauthorized:
                    md_log("蓝牙权限未打卡")
                    LQJargon.hiddenJargon(message: ("蓝牙权限未打开，请到设置中打开蓝牙权限".localization))
                    break
                case .poweredOff:
                    md_log("蓝牙未打卡")
                    LQJargon.hiddenJargon(message: "蓝牙未打开".localization)
                    break
                case .poweredOn:
//                    central.scanForPeripherals(withServices: nil, options: nil)
                    md_log("蓝牙可用")
                    LQJargon.hiddenJargon(message: "蓝牙已打开".localization)
                    break
                default: break
                }
    }
    
    
        func centralManager(_ central: CBCentralManager, didDiscover peripheral: CBPeripheral, advertisementData: [String : Any], rssi RSSI: NSNumber) {
    
            let name = peripheral.identifier.uuidString
                
            let c = dataList.contains(where: { obj in
                if obj.identifier.uuidString.elementsEqual(name) == YES {
                    obj.md.associatedObj = RSSI
                    return YES
                }
                return NO
            })
            if !c {
                md_log("蓝牙发现外设：\(name)")
                dataList.append(peripheral)
            }
            tableView.reloadData()
        }
    
    func centralManager(_ central: CBCentralManager, didConnect peripheral: CBPeripheral) {
        
        tableView.reloadData()

        md_log("蓝牙外设：\(peripheral.name!) 连接成功")
        LQJargon.hiddenJargon(message: "已连接外设".localization + " \(peripheral.name!)")
        
        connectPeripheral = peripheral
        
        peripheral.delegate = self
        peripheral.readRSSI()
        
//        let service = CBUUID(string: "0000ffe1-0000-1000-8000-00805f9b34fb")
//        peripheral.discoverServices([service])
        peripheral.discoverServices(nil)

        svc.modalPresentationStyle = .overFullScreen
        present(svc, animated: YES, completion: nil)

    }
    
    func centralManager(_ central: CBCentralManager, didFailToConnect peripheral: CBPeripheral, error: Error?) {
        md_log("蓝牙外设：\(peripheral.name!) 连接失败")
        LQJargon.hiddenJargon(message: "蓝牙外设".localization + "：\(peripheral.name!) " + "连接失败".localization)
    }
    
    func centralManager(_ central: CBCentralManager, didDisconnectPeripheral peripheral: CBPeripheral, error: Error?) {
        md_log("蓝牙外设：\(peripheral.name!) 取消连接")
        LQJargon.hiddenJargon(message: "蓝牙外设".localization + "：\(peripheral.name!) " + "取消连接".localization)
    }
    
}


extension ViewController : CBPeripheralDelegate {
    
    func peripheral(_ peripheral: CBPeripheral, didReadRSSI RSSI: NSNumber, error: Error?) {
        peripheral.md.associatedObj = RSSI
        tableView.reloadData()
    }
    
    func peripheral(_ peripheral: CBPeripheral, didDiscoverServices error: Error?) {
        guard let services = peripheral.services else {
            return
        }
        
        services.forEach { service in
            peripheral.discoverCharacteristics(nil, for: service)
            md_log("蓝牙外设：\(peripheral.name!) 发现服务")
        }
        
    }
    
    
    func peripheral(_ peripheral: CBPeripheral, didDiscoverCharacteristicsFor service: CBService, error: Error?) {
        md_log("蓝牙外设：\(peripheral.name!) 发现特征")
        
        if error != nil {
            md_log("特性错误")
        }
        
        service.characteristics?.forEach({ characteristic in
            md_log("蓝牙外设：\(peripheral.name!),发现 服务 \(service.uuid.uuidString) 的 特性 \(characteristic.uuid.uuidString)")
            
            peripheral.setNotifyValue(YES, for: characteristic)
            
        })
        
        if service.uuid.uuidString.elementsEqual(BELWriteDataServiceUUID) ||  service.uuid.uuidString.elementsEqual(BELWriteDataCharacteristicUUID) {
            startUHF()
        }
        
    }
    
    func peripheral(_ peripheral: CBPeripheral, didUpdateValueFor characteristic: CBCharacteristic, error: Error?) {
        if characteristic.value != nil {
            let value = md_string(form: characteristic.value!)
            instruction?.back.append(value)
            
            instruction?.validEnd()
           
            if instruction?.isEnd == YES {
                if instruction?.id == Instruction.ID.start {
                    LQJargon.hiddenJargon(message: "开启UHF成功".localization)
                    
                    let ins = Instruction(Instruction.ID.sessionModel0)
                    self.send(ins)
                    svc.sessionModelButton.setTitle("设置会话中...".localization, for:  .normal)
                } else if instruction?.id == Instruction.ID.sessionModel0 {
                    
                    LQJargon.hiddenJargon(message: "会话模式设置成功".localization)
                    svc.sessionModelButton.setTitle("会话模式0".localization, for:  .normal)
                    
                    let ins = Instruction(Instruction.ID.powerGet)
                    self.send(ins)

//                    if powerFirst {
//                        powerFirst = NO
//                        let ins = Instruction(Instruction.ID.power500)
//                        self.send(ins)
//                        svc.powerButton.setTitle("设置功率中...", for: .normal)
//
//
//                    }
                } else if instruction?.id == Instruction.ID.powerGet {
                    
                    svc.powerButton.isEnabled = YES
                    powerIsA4 = instruction!.getPowerIsA4()
                    
                    let ins = Instruction(powerIds.first!)
                    self.send(ins)
                    svc.powerButton.setTitle("设置功率中...".localization, for: .normal)

                } else if instruction?.id == Instruction.ID.sessionModel1 {
                    LQJargon.hiddenJargon(message: "会话模式设置成功".localization)
                    svc.sessionModelButton.setTitle("会话模式1".localization, for:  .normal)
                } else if instruction?.id == Instruction.ID.power500 {
                    LQJargon.hiddenJargon(message: "功率设置成功".localization)
                    svc.powerButton.setTitle("功率 500".localization, for: .normal)
                } else if instruction?.id == Instruction.ID.power1000 {
                    LQJargon.hiddenJargon(message: "功率设置成功".localization)
                    svc.powerButton.setTitle("功率 1000".localization, for: .normal)
                } else if instruction?.id == Instruction.ID.power1500 {
                    LQJargon.hiddenJargon(message: "功率设置成功".localization)
                    svc.powerButton.setTitle("功率 1500".localization, for: .normal)
                } else if instruction?.id == Instruction.ID.power2000 {
                    LQJargon.hiddenJargon(message: "功率设置成功".localization)
                    svc.powerButton.setTitle("功率 2000".localization, for: .normal)
                } else if instruction?.id == Instruction.ID.power2300 {
                    LQJargon.hiddenJargon(message: "功率设置成功".localization)
                    svc.powerButton.setTitle("功率 2300".localization, for: .normal)
                } else if instruction?.id == Instruction.ID.power2700 {
                    LQJargon.hiddenJargon(message: "功率设置成功".localization)
                    svc.powerButton.setTitle("功率 2700".localization, for: .normal)
                } else if instruction?.id == Instruction.ID.power3000 {
                    LQJargon.hiddenJargon(message: "功率设置成功".localization)
                    svc.powerButton.setTitle("功率 3000".localization, for: .normal)
                } else if instruction?.id == Instruction.ID.pdChina {
                    LQJargon.hiddenJargon(message: "频段设置成功".localization)
                    svc.pdButton.setTitle("频段：中国".localization, for: .normal)
                } else if instruction?.id == Instruction.ID.pdBeimei {
                    LQJargon.hiddenJargon(message: "频段设置成功".localization)
                    svc.pdButton.setTitle("频段：北美".localization, for: .normal)
                } else if instruction?.id == Instruction.ID.pdOzhou {
                    LQJargon.hiddenJargon(message: "频段设置成功".localization)
                    svc.pdButton.setTitle("频段：欧洲".localization, for: .normal)
                } else if instruction?.id == Instruction.ID.inventory {
                    if svc.isPanding == NO {
                        svc.inventoryButton.isEnabled = YES
                        svc.inventoryButton.backgroundColor = .lqsMain
//                        svc.inventoryLabel.text = "共读到\(svc.tagsMap.count)个标签"
                        return
                    }
                    let num = instruction!.readTagNum()
                    
                    if num > 0 {
                        let ins = Instruction(Instruction.ID.inventoryData)
                        self.send(ins)
                    } else if instruction!.needAgain() {
                        svc.inventoryButton.isEnabled = NO
                        svc.inventoryButton.backgroundColor = .lqsCancle
                        let ins = Instruction(Instruction.ID.inventory)
                        self.send(ins)
                    }
                    
                } else if instruction?.id == Instruction.ID.inventoryData {
                    
                    let tags = instruction!.readTags()
                    svc.append(tags)
                    
                    if svc.isPanding == NO {
                        svc.inventoryLabel.text = "共读到".localization + "\(svc.tagsMap.count)" + "个标签".localization
                        svc.inventoryButton.isEnabled = YES
                        svc.inventoryButton.backgroundColor = .lqsMain
                        return
                    }
                    
                    let ins = Instruction(Instruction.ID.inventory)
                    self.send(ins)
                }
                
                
            }
                        
            
            md_log("接受到 特性 \(characteristic.uuid.uuidString) 的数据 : \(value)")
        } else {
            md_log("接受到 特性 \(characteristic.uuid.uuidString) 的 空 数据")
        }
    }
    
    
//    func peripheral(_ peripheral: CBPeripheral, didWriteValueFor characteristic: CBCharacteristic, error: Error?) {
//
//        if characteristic.value != nil {
//            let value = md_string(form: characteristic.value!)
//            md_log("写入数据回调：特性 \(characteristic.uuid.uuidString) 的数据 : \(String(describing: value))")
//        } else {
//            md_log("写入数据回调：特性 \(characteristic.uuid.uuidString) 的 空 数据")
//        }
//    }
    
    
    func startUHF() {
        send(Instruction(Instruction.ID.start))
    }
    
    @discardableResult
    func send(_ ins:Instruction) -> Bool {
        if instruction?.isSending == YES {
            LQJargon.hiddenJargon(message: "失败，上一个指令发送中".localization)
            return NO
        }
        instruction  = ins
        let data = String.data(from: ins.id)
        if let services = connectPeripheral?.services {
            for s in services {
                if s.uuid.uuidString.elementsEqual(BELWriteDataServiceUUID) {
                    if  s.characteristics != nil {
                        for c in s.characteristics! {
                            connectPeripheral?.writeValue(data, for: c, type: .withResponse)
                            md_log("发送指令：\(ins.id)")
                            ins.send()
                            return YES
                        }
                    }
                }
            }
        }
        return NO
    }
    
    
}

extension ViewController : UITableViewDelegate, UITableViewDataSource {
    
    func tableView(_ tableView: UITableView, numberOfRowsInSection section: Int) -> Int {
        return dataList.count
    }

    func tableView(_ tableView: UITableView, cellForRowAt indexPath: IndexPath) -> UITableViewCell {
        
        var cell = tableView.dequeueReusableCell(withIdentifier: UITableViewCell.description())
        if cell == nil {
            cell = UITableViewCell(style: .subtitle, reuseIdentifier: UITableViewCell.description())
            cell!.selectionStyle = .none
            cell?.textLabel?.textColor = .lqsMainTitle
            cell?.detailTextLabel?.textColor = .lqsMainTitle
        }

        let peripheral = dataList[indexPath.row]

        cell?.textLabel?.text = (peripheral.name ?? "") + "(\(peripheral.identifier.uuidString))"

        if peripheral == self.connectPeripheral {
            if let s =  peripheral.md.associatedObj  as? NSNumber {
                cell?.detailTextLabel?.text = "信号强度".localization + "：\(s), " + "已连接".localization
            } else {
                cell?.detailTextLabel?.text = "已连接".localization
            }
        } else {
            if let s =  peripheral.md.associatedObj  as? NSNumber {
                cell?.detailTextLabel?.text = "信号强度".localization + "：\(s)"
            } else {
                cell?.detailTextLabel?.text = nil
            }
        }

        return cell!
    }
    
    
    func tableView(_ tableView: UITableView, didSelectRowAt indexPath: IndexPath) {
        let peripheral = dataList[indexPath.row]
        if peripheral == connectPeripheral {
            LQJargon.hiddenJargon(message: "该外设已被连接".localization)
            return
        }
        
        if connectPeripheral !=  nil {
            centralManager.cancelPeripheralConnection(connectPeripheral!)
        }
        
        centralManager.connect(peripheral, options: nil)

    }
    
    
    
}



class Instruction {
    let id : String
    
    var back : [String] = []
    
    var isSending = false
    var isEnd = NO

    init(_ id: String) {
        self.id = id
    }
    
    func send() {
        isSending = YES
    }
    
    func endReceived () {
        isSending = NO
        isEnd = YES
    }
    
    func validEnd() {
        if back.count == 0 {
            return
        }
        
        let first = back[0]
        
        let s = first.split(separator: ",")
        
        if s.count < 2 {
            md_log("字节验证：长度错误1")
            return
        }

        let hex = s[1].replacingOccurrences(of: " bytes = 0x", with: "").replacingOccurrences(of: "}", with: "")
        
        if hex.count < 4 {
            md_log("字节验证：长度错误2")
            return
        }
        
        let lenHex = (hex as NSString).substring(with: NSRange(location: 2, length: 2))
        
        
        var len : Int = 0
        for hex in back {
            let s = hex.split(separator: ",")
            if s.count < 1 {
                md_log("字节验证：长度错误3")
                return
            }
            let l = s.first?.replacingOccurrences(of: "{length = ", with: "")
            len += Int(l ?? "00") ?? 0
        }

        if len == md_int(lenHex) + 7 {
            endReceived()
        }
        
    }
    
    var isValid :Bool {
        if back.count == 0 {
            return NO
        }
        
        let first = back[0]
        
        let s = first.split(separator: ",")
        
        if s.count < 2 {
            md_log("有效验证：长度错误1")
            return NO
        }

//        let l = s.first?.replacingOccurrences(of: "{length = ", with: "")
        let hex = s[1].replacingOccurrences(of: " bytes = 0x", with: "").replacingOccurrences(of: "}", with: "")
        
        if hex.hasPrefix("ff") == NO {
            md_log("有效验证：错误开头")
            return NO
        }
        
        if hex.count < 6 {
            md_log("有效验证：长度错误2")
            return NO
        }
            
        
        let ffRang = NSRange(location: 4, length: 2)
        if (hex as NSString).substring(with: ffRang).elementsEqual((id as NSString).substring(with: ffRang)) == NO {
            md_log("有效验证：错误匹配")
            return NO
        }
        
        if hex.count < 10 {
            md_log("有效验证：长度错误2")
            return NO
        }
        
        let rang = NSRange(location: 6, length: 4)
        if (hex as NSString).substring(with: rang).elementsEqual("0000") == NO {
            md_log("有效验证：错误匹配0000")
            return NO
        }
        
        return YES
    }
    
    func getPowerIsA4()->Bool{
        if back.count == 0 {
            return NO
        }
        let first = back[0]
        let s = first.split(separator: ",")
        
        if s.count < 2 {
            md_log("A4验证：长度错误1")
            return NO
        }
        
        let hex = s[1].replacingOccurrences(of: " bytes = 0x", with: "").replacingOccurrences(of: "}", with: "")
        
        if hex.count > 20 {
            let rang = NSRange(location: 18, length: 2)
            let num = (hex as NSString).substring(with: rang)
            return md_int(num) == 164 //A4
        }

        
        return NO
    }
    
    
    func readTagNum() -> Int {
        
        if id.elementsEqual(ID.inventory) == NO {
            return 0
        }
        
        if back.count == 0 {
            return 0
        }
        
        let first = back[0]
        let s = first.split(separator: ",")
        if s.count < 2 {
            md_log("读取标签数：长度错误1")
            return 0
        }
        let hex = s[1].replacingOccurrences(of: " bytes = 0x", with: "").replacingOccurrences(of: "}", with: "")
        
        if hex.count < 8 {
            md_log("读取标签数：长度错误2")
            return 0
        }
        
        let rang = NSRange(location: hex.count - 6, length: 2)
        let num = (hex as NSString).substring(with: rang)
        return md_int(num)
    }
    
    
    func needAgain()->Bool {
        if id.elementsEqual(ID.inventory) == NO {
            return NO
        }
        
        if back.count == 0 {
            return NO
        }
        
        let first = back[0]
        let s = first.split(separator: ",")
        if s.count < 2 {
            md_log("重发pan指令：长度错误1")
            return YES
        }
        let hex = s[1].replacingOccurrences(of: " bytes = 0x", with: "").replacingOccurrences(of: "}", with: "")
        
        if hex.count < 10 {
            md_log("重发pan指令：长度错误2")
            return NO
        }
        let rang = NSRange(location: 6, length: 4)
        let num = (hex as NSString).substring(with: rang)
        return num.elementsEqual("0400")

    }
    
    func readTags() -> [BLETag] {
        if id.elementsEqual(ID.inventoryData) == NO {
            return []
        }
        
        if back.count == 0 {
            return []
        }
        
        var hex = ""
        
        for l in back {
            let s = l.split(separator: ",")
            if s.count < 2 {
                md_log("读取标签：长度错误1")
                return []
            }
            let h = s[1].replacingOccurrences(of: " bytes = 0x", with: "").replacingOccurrences(of: "}", with: "")
            hex += h
        }
        
        if hex.count < 18 {
            md_log("读取标签：长度错误2")
            return []
        }
        let numRange = NSRange(location: 16, length: 2)
        let num = md_int((hex as NSString).substring(with: numRange))
        
        var location = 18
        var tags : [BLETag] = []
        
        for _ in 0..<num {
//            let range = NSRange(location: location, length: 38)
//            if hex.count < (range.location + range.length) {
//                break
//            }
//            let cur = (hex as NSString).substring(with: range)
//            let tagNum = (cur as NSString).substring(with: NSRange(location: 0, length: 2))
//            let tagid = (cur as NSString).substring(with: NSRange(location: 10, length: 24)).uppercased()
//
//            let tag = BLETag(tagid)
//            tag.number = md_int(tagNum)
//
//            tags.append(tag)
//
//            location += 38
            
            let range = NSRange(location: location, length: hex.count - location)
            if hex.count < (range.location + range.length) {
                break
            }
            let cur = (hex as NSString).substring(with: range)
            if cur.count < 2 {
                md_log("读取标签：长度错误3")
                break
            }
            let tagNum = (cur as NSString).substring(with: NSRange(location: 0, length: 2))
            
            if cur.count < 6 {
                md_log("读取标签：长度错误4")
                break
            }
            let idLen_ = (cur as NSString).substring(with: NSRange(location: 2, length: 4))
            let idLen = (md_int(idLen_) - 32)/8 * 2
            
            if cur.count < (10 + idLen) {
                md_log("读取标签：长度错误5")
                break
            }
            let tagid = (cur as NSString).substring(with: NSRange(location: 10, length: idLen)).uppercased()

            let tag = BLETag(tagid)
            tag.number = md_int(tagNum)

            tags.append(tag)
            
            location += (14 + idLen)

        }
        
        
        return tags
    }
    
    
    struct ID {
        static let start = "FF00041D0B"
        static let sessionModel0 = "FF039B050000DCE8"
        static let sessionModel1 = "FF039B050001DCE9"

        static let powerGet = "FF00031D0C"
        static let power500 = "FF0891040101F401F401F4A388"
        static let power1000 = "FF0891040103E803E801F4CD3D"
        static let power1500 = "FF0891040105DC05DC01F42FB0"
        static let power2000 = "FF0891040107D007D001F41057"
        static let power2300 = "FF0891040108FC08FC01F4C2CE"
        static let power2700 = "FF089104010A8C0A8C01F4EF5B"
        static let power3000 = "FF089104010BB80BB801F4C56C"
        
        static let pdChina = "FF0197064BBB"
        static let pdBeimei = "FF0197014BBC"
        static let pdOzhou = "FF0197084BB5"
        
        static let inventory = "FF0522000003001438C8"
        static let inventoryData = "FF0329000100F522"
        
        
        static let lampGreenOpen = "FF0296010100DC"
        static let lampGreenClose = "FF0296010000DD"

        
        static let lampRedOpen = "FF0296020103DC"
        static let lampRedClose = "FF0296020003DD"
        
    }
    
}


class BLETag {
    var number : Int = 0
    var id : String
    var rank : Int = 0
    
    init(_ id: String) {
        self.id = id
    }
    
}
